<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Quotation;
use App\Models\QuotationItem;
use App\Models\QuotationItemDetail;
use App\Models\ProjectItem;
use App\Models\Project;
use Illuminate\Support\Facades\DB;

class QuotationController extends Controller
{
public function addToQuotation(Request $request)
{
    DB::beginTransaction();

    try {

        $project = Project::with([
            'items.measurement',
            'items.curtain',
            'items.wallpaper',
            'items.variant',
            'items.product'
        ])->findOrFail($request->project_id);

        // Get or create draft quotation
        $quotation = Quotation::firstOrCreate(
            ['project_id' => $project->id, 'status' => 'draft'],
            [
                'quotation_number' => 'QT-' . time(),
                'subtotal' => 0,
                'tax' => 0,
                'grand_total' => 0,
                'created_by' => auth()->id()
            ]
        );

        $addedCount = 0;
        $subtotal = $quotation->subtotal;
        $totalTax = $quotation->tax;

        foreach ($project->items as $item) {

            // ✅ Only non custom
            if ($item->is_custom ?? 0) {
                continue;
            }

            // ✅ Skip already added items
            $exists = QuotationItem::where('quotation_id', $quotation->id)
                        ->where('project_item_id', $item->id)
                        ->exists();

            if ($exists) {
                continue;
            }

            $price = 0;
            $quantity = $item->quantity;
            $unit = null;

            /* =====================================
               🎯 CURTAIN FABRIC CALCULATION
            ===================================== */
           if ($item->category == 'curtain' && $item->curtain && $item->measurement) {

    $width  = $item->measurement->width ?? 0;   // in cm
    $height = $item->measurement->height ?? 0;  // in cm

    $fabricWidth   = $item->curtain->fabric_width ?? 120; // available width
    $pricePerMeter = $item->curtain->price_per_meter ?? 0;

    if ($width <= 0 || $height <= 0 || $fabricWidth <= 0 || $pricePerMeter <= 0) {
        return response()->json([
            'success' => false,
            'message' => 'Curtain measurement or fabric data missing for item ID: ' . $item->id
        ]);
    }

    // Add stitching height (30 cm)
    $heightWithStitch = $height + 30;

    $panels = 0;

    // =========================================
    // 1️⃣ REGULAR CURTAINS
    // =========================================
    if (in_array(strtolower($item->curtain->type), ['eyelet','pleat','ripple','regular'])) {

        $panels = ceil(($width * 2.5) / $fabricWidth);
    }

    // =========================================
    // 2️⃣ ROMAN BLINDS
    // =========================================
    elseif (strtolower($item->curtain->type) == 'roman') {

        $widthWithFold = $width + 25;
        $panels = ceil($widthWithFold / $fabricWidth);
    }

    // Fallback safety
    else {
        $panels = ceil(($width * 2.5) / $fabricWidth);
    }

    // Total fabric in cm
    $totalCm = $heightWithStitch * $panels;

    // Convert to meters
    $totalMeters = $totalCm / 100;

    // Final price
    $price = $totalMeters * $pricePerMeter;

    $quantity = round($totalMeters, 2);
    $unit = 'meter';
}

            /* =====================================
               🎯 WALLPAPER CALCULATION
            ===================================== */
            elseif ($item->category == 'wallpaper' && $item->wallpaper) {

                $rollWidth = $item->wallpaper->roll_width ?? 0;
                $rollLength = $item->wallpaper->roll_length ?? 0;
                $wallWidth = $item->wallpaper->wall_width ?? 0;
                $wallHeight = $item->wallpaper->wall_height ?? 0;
                $pricePerRoll = $item->wallpaper->price_per_roll ?? 0;

                if ($rollWidth && $rollLength) {

                    $wallArea = $wallWidth * $wallHeight;
                    $rollArea = $rollWidth * $rollLength;

                    $rolls = ceil($wallArea / $rollArea);

                    $price = $rolls * $pricePerRoll;
                    $quantity = $rolls;
                    $unit = 'roll';
                }
            }

            /* =====================================
               🎯 NORMAL PRODUCT VARIANT PRICE
            ===================================== */
            else {

              if ($item->variant && $item->variant->offer_price > 0) {

    $unitPrice = $item->variant->offer_price;
    $price = $unitPrice * $quantity;
    $unit = $item->variant->unit;

} else {

    return response()->json([
        'success' => false,
        'message' => 'Offer price not found for variant ID: ' . $item->variant_id
    ]);
}
            }

            $product = $item->product;
            $gstPercent = $product->gst_percent ?? 0;
            $gstInclusive = $product->gst_inclusive ?? 0;

            $taxAmount = 0;

            if ($gstPercent > 0) {

                if ($gstInclusive) {
                    $base = $price / (1 + ($gstPercent/100));
                    $taxAmount = $price - $base;
                    $price = $base;
                } else {
                    $taxAmount = ($price * $gstPercent) / 100;
                }
            }

            $total = $price + $taxAmount;

            $quotationItem = QuotationItem::create([
                'quotation_id' => $quotation->id,
                'project_item_id' => $item->id,
                'category' => $item->category,
                'product_id' => $item->product_id,
                'variant_id' => $item->variant_id,
                'is_custom' => 0,
                'quantity' => $quantity,
                'unit' => $unit,
                'price' => round($price,2),
                'total' => round($total,2),
                'notes' => $item->notes
            ]);

            // Save all details including new custom fields
            $details = [];

            if ($item->curtain) {
                $details = $item->curtain->toArray();
            }

            if ($item->wallpaper) {
                $details = $item->wallpaper->toArray();
            }

            QuotationItemDetail::create([
                'quotation_item_id' => $quotationItem->id,
                'measurement' => json_encode($item->measurement),
                'details' => json_encode($details)
            ]);

            $subtotal += $price;
            $totalTax += $taxAmount;
            $addedCount++;
        }

        if ($addedCount == 0) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'All items already added to quotation!'
            ]);
        }

        $quotation->update([
            'subtotal' => round($subtotal,2),
            'tax' => round($totalTax,2),
            'grand_total' => round($subtotal + $totalTax,2)
        ]);

        $project->update(['status' => 'quoted']);

        DB::commit();

        return response()->json([
            'success' => true,
            'message' => $addedCount . ' items added to quotation successfully!'
        ]);

    } catch (\Exception $e) {

        DB::rollBack();

        return response()->json([
            'success' => false,
            'message' => $e->getMessage()
        ]);
    }
}
}